﻿type
  // Представление куба: массив из 6 граней
  // Индексы: 0-передняя, 1-задняя, 2-левая, 3-правая, 4-верхняя, 5-нижняя
  TCube = array[0..5] of integer;
  
  // Вращение куба как перестановка граней
  TRotation = array[0..5] of integer;

function Factorial(n: integer): int64;
begin
  Result := 1;
  for var i := 2 to n do
    Result := Result * i;
end;

// Применяет вращение к раскраске
function ApplyRotation(const coloring: TCube; const rotation: TRotation): TCube;
begin
  for var i := 0 to 5 do
    Result[i] := coloring[rotation[i]];
end;

// Проверяет, одинаковая ли раскраска
function SameColoring(const a, b: TCube): boolean;
begin
  for var i := 0 to 5 do
    if a[i] <> b[i] then
      Exit(false);
  Result := true;
end;

// Создаёт все 6! = 720 раскрасок (без учёта симметрии)
function GenerateAllColorings(): List<TCube>;
begin
  Result := new List<TCube>;
  var colors := Arr(1, 2, 3, 4, 5, 6); // 6 разных цветов
  
  // Генерация всех перестановок 6 цветов
  var perm := colors.ToArray;
  
  repeat
    var coloring: TCube;
    for var i := 0 to 5 do
      coloring[i] := perm[i];
    Result.Add(coloring);
  until not NextPermutation(perm);
end;

// Основная функция подсчёта различных раскрасок
function CountDistinctColorings(): integer;
begin
  // Все 24 вращения куба как перестановки граней
  // Каждое вращение - массив, где rotation[i] показывает,
  // на какую грань перемещается грань i
  
  var rotations: array of TRotation = (
    // Тождественное преобразование (1 вращение)
    (0, 1, 2, 3, 4, 5),
    
    // Вращения вокруг осей через центры противоположных граней (по 3 на каждую ось)
    // Ось через переднюю-заднюю грань
    (0, 1, 4, 5, 3, 2),  // 90° по часовой
    (0, 1, 5, 4, 2, 3),  // 180°
    (0, 1, 3, 2, 5, 4),  // 270°
    
    // Ось через левую-правую грань
    (5, 4, 2, 3, 0, 1),  // 90°
    (1, 0, 2, 3, 5, 4),  // 180°
    (4, 5, 2, 3, 1, 0),  // 270°
    
    // Ось через верхнюю-нижнюю грань
    (2, 3, 1, 0, 4, 5),  // 90°
    (1, 0, 3, 2, 4, 5),  // 180°
    (3, 2, 0, 1, 4, 5),  // 270°,
    
    // Вращения вокруг осей через противоположные вершины (по 2 на каждую ось)
    // Ось через вершины 0-2-5 и 1-3-4 (условно)
    (2, 3, 5, 4, 0, 1),  // 120°
    (5, 4, 1, 0, 3, 2),  // 240°
    
    // Ось через вершины 0-3-5 и 1-2-4
    (3, 2, 0, 1, 5, 4),  // 120°
    (5, 4, 3, 2, 1, 0),  // 240°,
    
    // Ось через вершины 0-1-4 и 2-3-5
    (1, 0, 3, 2, 5, 4),  // 120°
    (4, 5, 2, 3, 0, 1),  // 240°,
    
    // Ось через вершины 0-1-5 и 2-3-4
    (1, 0, 2, 3, 4, 5),  // 120°
    (4, 5, 3, 2, 1, 0),  // 240°,
    
    // Вращения вокруг осей через середины противоположных рёбер (по 1 на каждую ось)
    // Ось через середины рёбер 0-2 и 1-3
    (2, 3, 1, 0, 5, 4),
    
    // Ось через середины рёбер 0-3 и 1-2
    (3, 2, 0, 1, 5, 4),
    
    // Ось через середины рёбер 0-4 и 1-5
    (4, 5, 3, 2, 1, 0),
    
    // Ось через середины рёбер 0-5 и 1-4
    (5, 4, 2, 3, 0, 1),
    
    // Ось через середины рёбер 2-4 и 3-5
    (0, 1, 4, 5, 3, 2),
    
    // Ось через середины рёбер 2-5 и 3-4
    (0, 1, 5, 4, 2, 3)
  );
  
  // Генерируем все раскраски
  var allColorings := GenerateAllColorings();
  WriteLn('Всего возможных раскрасок (без учёта симметрии): ', allColorings.Count);
  
  // Применяем лемму Бернсайда
  var fixedCounts := new List<integer>;
  
  for var rotIdx := 0 to rotations.High do
  begin
    var fixedForThisRotation := 0;
    
    // Для каждой раскраски проверяем, остаётся ли она неизменной
    // при данном вращении
    foreach var coloring in allColorings do
    begin
      var rotated := ApplyRotation(coloring, rotations[rotIdx]);
      if SameColoring(coloring, rotated) then
        fixedForThisRotation += 1;
    end;
    
    fixedCounts.Add(fixedForThisRotation);
  end;
  
  // Суммируем и делим на размер группы
  var sumFixed := fixedCounts.Sum();
  Result := sumFixed div rotations.Length;
  
  // Выводим информацию о фиксированных раскрасках для каждого вращения
  WriteLn(NewLine + 'Фиксированные раскраски для каждого вращения:');
  for var i := 0 to rotations.High do
    WriteLn($'Вращение {i+1}: {fixedCounts[i]}');
end;

// Альтернативное решение через формулу леммы Бернсайда
function CountByBurnsideFormula(): integer;
begin
  // Для 6 различных цветов на кубе:
  // Вращения группируются по типам:
  
  // 1) Тождественное преобразование (1 вращение)
  //    Фиксирует все 6! = 720 раскрасок
  
  // 2) Вращения на 90°/270° вокруг осей через центры граней (6×2 = 12 вращений)
  //    Ось фиксирует 2 противоположные грани, остальные 4 циклически меняются
  //    Фиксирует только раскраски, где все 4 циклические грани одного цвета,
  //    но у нас все цвета разные → 0 фиксированных
  
  // 3) Вращения на 180° вокруг осей через центры граней (3 вращения)
  //    Фиксирует пары противоположных граней
  //    Для 6 разных цветов: 0 фиксированных
  
  // 4) Вращения на 120°/240° вокруг осей через вершины (8 вращений)
  //    Фиксируют 2 противоположные вершины, остальные циклически
  //    Для 6 разных цветов: 0 фиксированных
  
  // 5) Вращения на 180° вокруг осей через середины рёбер (6 вращений)
  //    Меняют местами пары граней
  //    Для 6 разных цветов: 0 фиксированных
  
  // Итого по лемме Бернсайда:
  // (720 + 12×0 + 3×0 + 8×0 + 6×0) / 24 = 720 / 24 = 30
  
  Result := 720 div 24;
end;

// Визуализация одного примера раскраски
procedure PrintColoringExample();
begin
  WriteLn(NewLine + '=' * 50);
  WriteLn('ПРИМЕР ОДНОЙ ИЗ 30 РАЗЛИЧНЫХ РАСКРАСОК:');
  WriteLn('=' * 50);
  
  // Представим куб развёрткой:
  //      [4]
  // [2] [0] [3] [1]
  //      [5]
  
  var example: TCube = (1, 2, 3, 4, 5, 6); // Простая нумерация
  
  WriteLn('Развёртка куба:');
  WriteLn('       ', example[4]);        // Верхняя грань
  WriteLn(' ', example[2], '   ', example[0], '   ', example[3], '   ', example[1]); // Боковые
  WriteLn('       ', example[5]);        // Нижняя грань
  WriteLn;
  
  WriteLn('Цвета граней:');
  WriteLn('Передняя (0): ', example[0]);
  WriteLn('Задняя   (1): ', example[1]);
  WriteLn('Левая    (2): ', example[2]);
  WriteLn('Правая   (3): ', example[3]);
  WriteLn('Верхняя  (4): ', example[4]);
  WriteLn('Нижняя   (5): ', example[5]);
end;
             

begin
   Writeln(' Cube Coloring Problem');    
   WriteLn(' ');
   WriteLn;
  WriteLn('=' * 60);
  WriteLn('ЗАДАЧА О РАСКРАСКЕ КУБА 6 РАЗНЫМИ ЦВЕТАМИ');
  WriteLn('=' * 60);
  WriteLn;
  WriteLn('Условие: сколькими способами можно раскрасить грани куба');
  WriteLn('шестью различными цветами? (Раскраски, совпадающие при');
  WriteLn('вращении куба, считаются одинаковыми.)');
  WriteLn;
  
  // Способ 1: прямое применение леммы Бернсайда
  WriteLn('СПОСОБ 1: Прямой подсчёт через лемму Бернсайда');
  WriteLn('-' * 60);
  
  var timer := new System.Diagnostics.Stopwatch;
  timer.Start;
  
  var count1 := CountDistinctColorings();
  
  timer.Stop;
  WriteLn(NewLine + 'Результат: ', count1, ' различных раскрасок');
  WriteLn('Время вычисления: ', timer.ElapsedMilliseconds, ' мс');
  
  // Способ 2: по формуле
  WriteLn(NewLine + 'СПОСОБ 2: По формуле леммы Бернсайда');
  WriteLn('-' * 60);
  
  var count2 := CountByBurnsideFormula();
  WriteLn('Результат: ', count2, ' различных раскрасок');
  
  // Проверка совпадения результатов
  WriteLn(NewLine + 'Проверка: оба метода дают ', 
    count1 = count2 ? 'одинаковый результат' : 'разные результаты');
  
  // Вывод математического обоснования
  WriteLn(NewLine + '=' * 60);
  WriteLn('МАТЕМАТИЧЕСКОЕ ОБОСНОВАНИЕ:');
  WriteLn('=' * 60);
  
  WriteLn('1. Всего перестановок 6 цветов: 6! = ', Factorial(6));
  WriteLn('2. Группа вращений куба: 24 элемента');
  WriteLn('3. По лемме Бернсайда:');
  WriteLn('   N = (1/24) * Σ|Fix(g)|, где g ∈ G');
  WriteLn('4. Для 6 РАЗНЫХ цветов:');
  WriteLn('   - Тождественное вращение: фиксирует все 6! = 720 раскрасок');
  WriteLn('   - Все остальные вращения: 0 фиксированных раскрасок');
  WriteLn('     (т.к. при любом нетривиальном вращении хотя бы две грани');
  WriteLn('      меняются местами, а цвета все разные)');
  WriteLn('5. Итого: N = (720 + 23×0) / 24 = 720 / 24 = 30');
  
  // Пример раскраски
  PrintColoringExample();
  
  // Дополнительно: что если цвета могут повторяться?
  WriteLn(NewLine + '=' * 60);
  WriteLn('ДЛЯ СПРАВКИ: ЕСЛИ БЫ ЦВЕТА МОГЛИ ПОВТОРЯТЬСЯ...');
  WriteLn('=' * 60);
  WriteLn('Если бы было k цветов (не обязательно разных):');
  WriteLn('По лемме Бернсайда для раскраски граней куба в k цветов:');
  WriteLn('N(k) = (k⁶ + 3k⁴ + 12k³ + 8k²) / 24');
  WriteLn;
  WriteLn('Для k=6 (6 доступных цветов, могут повторяться):');
  var k := 6;
  var total := (Power(k,6) + 3*Power(k,4) + 12*Power(k,3) + 8*Power(k,2)) / 24;
  WriteLn($'N({k}) = ({k}⁶ + 3×{k}⁴ + 12×{k}³ + 8×{k}²) / 24 = {total}');
  WriteLn('Но в нашей задаче цвета РАЗНЫЕ, поэтому ответ: 30');        
   Writeln; 
   
  // Ответ для куба с 6 РАЗНЫМИ цветами
  WriteLn('Ответ: 30 способов');
  WriteLn;
  WriteLn('Объяснение:');
  WriteLn('1. Без учёта симметрии: 6! = 720 раскрасок');
  WriteLn('2. Группа вращений куба: 24 элемента');
  WriteLn('3. По лемме Бернсайда: (720 + 0 + ... + 0) / 24 = 30');
  WriteLn('   (только тождественное вращение фиксирует раскраски,');
  WriteLn('    остальные 23 вращения дают 0 фиксированных раскрасок');
  WriteLn('    т.к. все цвета разные)');
end.
